********************************************************************************
******                        Read in the data                        **********

clear
cd "C:\Users\worri\Dropbox\Poverty\COVID19"
use Monthly_Poverty_0520.dta 

gen yr_month=year*100+month   
keep if yr_month>=202001
** replace asecflag 2 => asecflag . 
replace asecflag=. if asecflag==2 
** drop irrelevant variables
drop month hflag qinc* qoinc* cpsid   

********************************************************************************
******   Construct a family unit variable for Poverty measures, famnum    ******
** For the poverty measure, a primary family and realted family is considered as one family

/* Assign 1 to primary+realted families */
gen famnum = ftype==1|ftype==3 

/* Assign 2-4 to unrelated subfamilies */ 
sort yr_month asecflag serial ftype pernum
* Each subfamily has a reference person  (famrel==1)
//tab famrel if ftype==4
* Calculate # subfamilies in a household (Max 3 in this sample period)
bysort yr_month asecflag serial: egen tot_subfam = sum(ftype==4 & famrel==1)
tab tot_subfam if ftype==4 
sort yr_month asecflag serial ftype famrel
bysort yr_month asecflag serial ftype: gen subfamnum = sum(famrel==1)
tab subfamnum if ftype==4 
* Assign 2-4 to the 1st-3rd unrelated subfamily
replace famnum= subfamnum+1 if ftype==4 
drop subfamnum 


/* Assign 5-20 to non-family members (ftype=1,2 or famrel=0) */
* Calculate # non-family members within a household (Max 16 in this sample period)
bysort yr_month asecflag serial: gen non_fam = sum(famrel==0) 
tab non_fam 
* Assign 5-20 to the 1st-20th non-family member. 
replace famnum= non_fam+4 if famrel==0  
drop non_fam

* Construct demographic variables related to the poverty threshold 
* 1) family size
bysort yr_month asecflag serial famnum: gen fam_size = _N
* 2) related children under age 18 (excluding HH head or spouse)
bysort yr_month asecflag serial famnum: egen nchild = sum(age<18 & relate>=301)
* 3) the head is elderly (>=65)
bysort yr_month asecflag serial famnum: egen head_elderly = max(age>=65 & relate==101)
* 4) a few cases where all members are children -> change to one adult w/ two children 
replace nchild=nchild-1 if fam_size==nchild


* Combine with the Census poverty threshold data
merge m:1 year fam_size nchild head_elderly using pov_thresh
keep if _merge==3
drop _merge


* merge with the imputed family income data
merge m:1 yr_month asecflag serial using impute
keep if yr_month>=202001
drop _merge

********************************************************************************
* demographic variables 
gen male = sex==1
drop sex  

** merge with the state UI parameter data
merge m:1 yr_month statefip using C:\Users\worri\Dropbox\Poverty\COVID19\st_UI.dta 
drop _merge

********************************************************************************
** estimate population at the national and the state-group level
bysort yr_month: egen totwgt = sum(wtfinl) 
bysort yr_month stategr: egen stgr_totwgt = sum(wtfinl) 

*** Restrict the sample to the householder's family 
bysort yr_month asecflag serial famnum: egen hfam = sum(relate==101) 
keep if hfam==1

*** Restrict the sample to individuals with non-imputed income 
gen miss_inc = (faminc>=995 & faminc<=999)|(qfaminc>0 & qfaminc!=.)
keep if miss_inc!=1




********************************************************************************
*******************         Imputation of EIP                  *****************
********************************************************************************


** step 1. identify sub families (every subfamily has a reference person(famrel==1))
tab famrel
* number of adults (for a family has no adult, I treat the head as an adult)
bysort yr_month asecflag serial famnum: egen num_adult = sum(age>=18|famrel==1) 
* number of subfamilies within a householder's family 
** person who is not living in a family (famrel==0) is a seprate famnum => one subfamilly
** each subfamily has a reference person (famrel==1) => # subfamilies equals # famrel==1 

bysort yr_month asecflag serial famnum: egen num_fam = sum(famrel==0|famrel==1)
sort yr_month asecflag serial famnum ftype famrel
bysort yr_month asecflag serial famnum: gen subfam = sum(famrel==0|famrel==1)
* this code leads to a misclassification of a family: for a family with two related families with a couple in each related family, both spouses are assigned to the second related family. 
* I manually assign one of spouses to the first related family.
bysort yr_month asecflag serial famnum subfam: gen nspouse = sum(famrel==2)            // head's tax unit
replace subfam=2 if subfam==3 & famrel==2 & nspouse==2
* number of adults in each subfam (the family head is treated as an adult regardless of age)
bysort yr_month asecflag serial famnum subfam: egen n_adult = sum(age>=18|famrel==0|famrel==1)       

** Calculate income at the subfamily level: 
** Rule: each subfamily's contribution to the total family income is proportional to the number of adults in a family
sum num_adult n_adult
gen sfam_inc = ifam_inc/(num_adult)*n_adult


********************************************************************************
** step 2. impute tax filing units
** Rules
** A) each subfamily (primary/related subfamily) is as a separate tax unit
** within each subfamily:
** B) a spouse of the family head: head's tax unit
** C) a person who is not a spouse of the family head and age above 23: separate tax unit
** D) a person who is not a spouse of the family head and age 23 or below: head's tax unit as dependent 
** E) a tax unit's contribution to the total subfamily income is proportional to the number of adults in the taxunit


* identify a separate tax unit within a subfamily (a person age 24 or above who is not a spouse of the family head)
gen tunit = ((famrel==3|famrel==4) & age>=24)
sort yr_month asecflag serial famnum subfam tunit
bysort yr_month asecflag serial famnum subfam: gen taxunit = sum(tunit)
* currently, taxunit takes a value of 0 for the head's unit. I change the lowest value to 1 
replace taxunit = taxunit+1
tab taxunit
* number of individual in a tax unit
bysort yr_month asecflag serial famnum subfam taxunit: gen tu_size = _N
* number of adults in a tax unit (head is treated as adult regardless of age)
bysort yr_month asecflag serial famnum subfam taxunit: egen tu_n_adult = sum(age>=18|famrel==0|famrel==1) 
* number of non-dependents in a tax unit
bysort yr_month asecflag serial famnum subfam taxunit: egen tu_n_nodep = sum(famrel==0|famrel==1|famrel==2|age>=24)
* number of dependents in a tax unit
gen tu_n_dep = tu_size-tu_n_nodep
* double check if the number of non-dependents in a tax unit is 2 of less
tab tu_n_nodep


** Calculate income at the tax unit level
** Rule: a tax unit's contribution to the total subfamily income is proportional to the number of adults in the taxunit
sum num_adult n_adult tu_n_adult
gen tfam_inc = (sfam_inc/n_adult)*tu_n_adult 
order yr_month asecflag serial famnum subfam taxunit ifam_inc sfam_inc tfam_inc

********************************************************************************
** check how many individuals living in households with only one family
preserve
replace wtfinl = round(wtfinl)
tab subfam [w=wtfinl]  // 3.6% ind. living in a related subfamily
bysort yr_month asecflag serial famnum: egen nsubfam = max(subfam)
sort yr_month asecflag serial famnum relate
bysort yr_month asecflag serial famnum: keep if _n==1
tab nsubfam [w=wtfinl]
restore
** check how many individuals living in a family with only one taxunit
preserve
replace wtfinl = round(wtfinl)
tab taxunit [w=wtfinl]  // 3.6% ind. living in a related subfamily
bysort yr_month asecflag serial famnum subfam: egen ntaxunit = max(taxunit)
sort yr_month asecflag serial famnum subfam relate
bysort yr_month asecflag serial famnum subfam: keep if _n==1
tab ntaxunit [w=wtfinl]
restore


********************************************************************************
** step 3. Impute EIP 
gen EIP = 0
replace EIP = 1200*tu_n_nodep+500*tu_n_dep if tfam_inc<=75000*tu_n_nodep
replace EIP = 1200*tu_n_nodep+500*tu_n_dep-0.05*(tfam_inc-75000*tu_n_nodep) if tfam_inc>75000*tu_n_nodep & tfam_inc<99000*tu_n_nodep
* convert the tax unit level EIP to the per capita EIP
replace EIP = EIP/tu_size
gen EIP_elig = EIP>0


** choose a random sample of EIP eligibles so that the total dollars of benefits that we impute matches the admin. data   
preserve 
* keep one obs per HH (head)
sort yr_month asecflag serial relate
bysort yr_month asecflag serial: keep if _n==1
set seed 1266
bysort yr_month asecflag: gen rnum = floor(100*runiform()+1)
keep yr_month asecflag serial rnum
tempfile EIP_rnum
save `EIP_rnum', replace
restore
merge m:1 yr_month asecflag serial using `EIP_rnum'
drop _merge
replace EIP = 0 if yr_month<=202003
replace EIP = 0 if rnum>59 & yr_month==202004
replace EIP = 0 if rnum>93 & yr_month==202005
drop rnum


********************************************************************************
******************         Imputation of UI Benefits         *******************
********************************************************************************
** Step 1. combine with the UI benefit calculator 
preserve
import delimited using "State UI benefit calculations", clear
drop if statefip==.
rename ïstate state
label var state "State"
tempfile UIrule
save `UIrule'
restore

*merge state rules data with CPS 
merge m:1 statefip using `UIrule'
drop _merge

** Step 2. Define the PUA and regular UI eligibles 
* 1) the unemployed who were previously self-employed  
gen PUA_unemp = (empstat==21|empstat==22) & (classwkr==13|classwkr==14) 
* 2) the employed who are absent from work due to health reasons, family responsibilities, childcare problems, and other reasons
* 3) those who want a job but did not look for work for 4 weeks because: 1) they believe no work is available in their area of expertise, they could not find a job, of family responsibilities, they cannot arrange childcare, or of other reasons.
gen PUA_nounemp = (((whyabsnt>=6 & whyabsnt<=8)|whyabsnt==15)|(wnlook==1|wnlook==2|wnlook==6|wnlook==7|wnlook==11)) 
* PUA eligibles
gen PUA_elig = (PUA_unemp==1|PUA_nounemp==1)
* Regular UI eligibles 
gen RUI_elig = (empstat==21|empstat==22) & PUA_elig==0

 
* Impute the duration of unemployment for those who are not unemployed (assign the average unemployment duration among the unemployed) 
bysort yr_month: sum durunemp [w=wtfinl] if durunemp>=0 & durunemp!=999
replace durunemp = 8 if PUA_nounemp==1 & yr_month==202004
replace durunemp = 11 if PUA_nounemp==1 & yr_month==202005
replace durunemp = 13 if PUA_nounemp==1 & yr_month==202006
* indicator for UI eiglbie 
gen UI_elig = durunemp>=0 & durunemp!=999
bysort yr_month: sum PUA_elig RUI_elig [w=wtfinl] if UI_elig==1
* I will change the indicators for UI and PUA eligibles below. So I keep the original indicators using a different name  
gen OUI_elig = UI_elig
gen OPUA_elig = PUA_elig

** Step 3. Imputing pre-separation earnings
** create variables for the prediction equation
* ORG sample indicator 
gen earnsample = mish==4|mish==8

* age
gen age_gr = 0
replace age_gr = 1 if age<18
replace age_gr = 2 if age>=18 & age<30
replace age_gr = 3 if age>=31 & age<45
replace age_gr = 4 if age>=46 & age<65
replace age_gr = 5 if age>=65

* edu
gen educa = 0
replace educa=1 if educ<73
replace educa=2 if educ==73   
replace educa=3 if educ>=74 & educ<111  
replace educa=4 if educ>=111 & educ!=.

** race/ethnicity
gen racethn = 0
replace racethn = 1 if hispan==0 & race==100 // white
replace racethn = 2 if hispan==0 & race==200 // black
replace racethn = 3 if hispan==0 & race!=100 & race!=200  // other
replace racethn = 4 if hispan!=0  // hispan

** occupation (9)
gen occ_ma = (occ>=10 & occ<=3655)    // managerial, professional, technical
gen occ_se = (occ>=3700 & occ<=4655)  // service 
gen occ_sa =  (occ>=4700 & occ<=4965)   // Sales
gen occ_of = (occ>=5000 & occ<=5940)   // Office/Admin. Support
gen occ_fa = (occ>=6005 & occ<=6130)     // Farming, Fishing, and Forestry (Natural resources)
gen occ_co = (occ>=6200 & occ<=7640)   // Construction/Repair
gen occ_pr = (occ>=7700 & occ<=9760)   // Production/Transportation
gen occ_mi = (occ>=9800 & occ<=9840) // Military
gen occ_no = occ==0                  // Nonemployed

* Industry Recodes (10)
gen ind_ag = (ind>=170 & ind<=490)   //agriculture, mining 
gen ind_co = (ind>=770 & ind<=3990)  //construction, manufacturing 
gen ind_wh = (ind>=4070 & ind<=5790) //retale trade
gen ind_tr = (ind>=6070 & ind<=6390)|(ind>=570 & ind<=690) //Transportation and utilities
gen ind_in = (ind>=6470 & ind<=6780) //information
gen ind_fi = (ind>=6870 & ind<=7190) //finance
gen ind_pr = (ind>=7270 & ind<=7790) //Professional and business services
gen ind_ed = (ind>=7860 & ind<=9590) //other service, public admin
gen ind_ar = (ind>=9670 & ind<=9870) //Armed Forces
gen ind_no = (ind==9920)             //nonemployed


** Predict earnings
** recode missing earnings
replace earnweek=. if earnweek==9999.99
reg earnweek age male i.educa i.racethn ind_* occ_* i.statefip i.yr_month 
predict pred_earn
** Applying adjustment factors for PUA and RUI eilgibles (codes for estimating the adjustment factor is at the bottom)
* scale factor for RUI eligibles: 0.87
replace pred_earn=pred_earn*0.875 if RUI_elig==1
* scale factor for PUA eligibles: 0.83
replace pred_earn=pred_earn*0.836 if PUA_elig==1
* bottom-coding weekly earnings at $36.25 (5*7.25)
replace pred_earn=36.25 if pred_earn<36.25 
* UI_elig 
replace pred_earn=0 if OUI_elig==0

* convert to quarterly earnings
gen qtrearn=pred_earn*13

********************************************************************************
*** Restrict the sample to individuals in the 1st or 5th month interview
keep if mish==1|mish==5


********************************************************************************
**  Construct Fixed Demographc Weight (Using the demographic composition in Jan & Feb 2020)
merge 1:1 yr_month asecflag serial pernum using fixedwgt.dta
** for March 2020 onward, use the fixed demographic weight
replace wtfinl = fixedwgt if yr_month>=202003



********************************************************************************
** Step 4. Impute UI benefits 

* estimate benefits for regular UI recipients 
gen estben=min(maxben,max(minben,qtrearn*replacement_rate))
label var estben "Estimated Weekly UI benefit"
* estimate benefits for PUA UI recipients 
replace estben = max(estben,pua_min) if PUA_elig==1
* zero benefits to those who are not eligible for UI
replace estben = 0 if UI_elig==0


*********************************************************************************
** Step 5. Select a subset of PUA eligibles so that the fraction of potential recipients in PUA/UI eligible groups matches the distribution of these types among actual recipients in the admin. data (11%, 32%, and 39% in April, May, and June, respectively)  
* calculate the number of UI eligibles 
bysort yr_month asecflag UI_elig: gen N_UIelig = _N
* assign numbers 1-N to PUA eligibles where N refers to the total number of PUA eligibles
sort yr_month asecflag UI_elig PUA_elig serial famnum pernum
set seed 126
bysort yr_month asecflag UI_elig PUA_elig: gen rnum = floor(1000*runiform()+1)

* the original fraction of potential recipients in PUA/UI eligible groups
bysort yr_month: sum PUA_elig RUI_elig [w=wtfinl] if UI_elig==1

gen PUA_cap = 0
replace PUA_cap = 850 if yr_month==202004
replace PUA_cap = 460 if yr_month==202005
replace PUA_cap = 12 if yr_month==202006

* reduce the number of potential PUA recipients 
replace UI_elig=0 if PUA_elig==1 & rnum<=PUA_cap
replace durunemp=999 if PUA_elig==1 & rnum<=PUA_cap
replace PUA_elig=0 if PUA_elig==1 & rnum<=PUA_cap
*/
* the new fraction of potential recipients in PUA/UI eligible groups (11.6% 32.2%, 39.2%)
bysort yr_month: sum PUA_elig RUI_elig [w=wtfinl] if yr_month>=202004 & UI_elig==1 


********************************************************************************
*** The cumulative UI benefits: WBA*#weeks unemployed since the week the payment started   
gen PUC = 0
gen PUA = 0
gen RUI = 0
** PUC 
replace PUC = 600*min(durunemp,4) if durunemp!=999 & yr_month==202004
replace PUC = 600*min(durunemp,8) if durunemp!=999 & yr_month==202005
replace PUC = 600*min(durunemp,12) if durunemp!=999 & yr_month==202006

** PUA  
replace PUA = estben*min(durunemp,13) if durunemp!=999 & PUA_elig==1 & yr_month==202004
replace PUA = estben*min(durunemp,17) if durunemp!=999 & PUA_elig==1 & yr_month==202005
replace PUA = estben*min(durunemp,21) if durunemp!=999 & PUA_elig==1 & yr_month==202006

** Regular UI 
replace RUI = estben*min(durunemp,4) if durunemp!=999 & RUI_elig==1 & yr_month==202004
replace RUI = estben*min(durunemp,8) if durunemp!=999 & RUI_elig==1 & yr_month==202005
replace RUI = estben*min(durunemp,12) if durunemp!=999 & RUI_elig==1 & yr_month==202006

* for Oregon, there is one week waiting period for UI 
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=3 & RUI_elig==1 & yr_month==202004 & statefip==41
replace RUI = estben*3 if durunemp>=4 & durunemp!=999 & RUI_elig==1 & yr_month==202004 & statefip==41
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=7 & RUI_elig==1 & yr_month==202005 & statefip==41
replace RUI = estben*7 if durunemp>=8 & durunemp!=999 & RUI_elig==1 & yr_month==202005 & statefip==41
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=11 & RUI_elig==1 & yr_month==202006 & statefip==41
replace RUI = estben*11 if durunemp>=12 & durunemp!=999 & RUI_elig==1 & yr_month==202006 & statefip==41


********************************************************************************
** Step 5. Match the simulated total UI payments to the actual total payment at the state-group level 
/*
preserve 
* calculate population in each state-group
bysort yr_month stategr: egen stgr_wgt = sum(wtfinl) 

* for each UI group (PUA and RUI), assign a random number of 1-100 
sort yr_month asecflag stategr UI_elig PUA_elig serial famnum pernum
set seed 126
bysort yr_month asecflag stategr UI_elig PUA_elig: gen pct = floor(100*runiform()+1)

forvalues ym = 202004/202006 {
	forvalues stgr = 1/3 { 
		gen r_`ym'`stgr' = 0
		gen tot_`ym'`stgr' = 0
	}
}

* Find the percent x, such that when x% of UI eligibles receive the benefits, the simulated payment matches the actual payment for a given state group and month  
forvalues ym = 202004/202006 {
	forvalues stgr = 1/3 {
		forvalues r=1/100 {
			di `r'
			replace PUC=0 if yr_month==`ym' & stategr==`stgr' & pct<`r' 
			replace PUA=0 if yr_month==`ym' & stategr==`stgr' & pct<`r'
			replace RUI=0 if yr_month==`ym' & stategr==`stgr' & pct<`r'
			gen allUI = RUI+PUC+PUA
			bysort yr_month asecflag serial famnum: egen fam_allUI = sum(allUI) 
			gen pc_allUI = fam_allUI/fam_size
			egen totallUI = sum((stgr_totwgt/stgr_wgt)*wtfinl*pc_allUI) if yr_month==`ym' & stategr==`stgr'
			sum totallUI
			local sim_totUI = r(mean)
            sum stgr_ccap if yr_month==`ym' & stategr==`stgr'
			local act_totUI = r(mean)
			replace r_`ym'`stgr'=`r'
			replace tot_`ym'`stgr'=`sim_totUI'
			drop allUI fam_allUI pc_allUI totallUI
			if `sim_totUI'<=`act_totUI' continue, break
			
		}
	}			
}
sum r_*
restore 
*/
********************************************************************************
** choose a random sample of potential UI eligibles so that the total dollars of benefits that we impute matches the admin. data   

sort yr_month asecflag stategr UI_elig PUA_elig serial famnum pernum
set seed 126
bysort yr_month asecflag stategr UI_elig PUA_elig: gen pct = floor(100*runiform()+1)
gen receive = 0
replace receive = 1 if yr_month==202004 & stategr==1 & pct>=69
replace receive = 1 if yr_month==202004 & stategr==2 & pct>=57
replace receive = 1 if yr_month==202004 & stategr==3 & pct>=56

replace receive = 1 if yr_month==202005 & stategr==1 & pct>=55
replace receive = 1 if yr_month==202005 & stategr==2 & pct>=31
replace receive = 1 if yr_month==202005 & stategr==3 & pct>=20

replace receive = 1 if yr_month==202006 & stategr==1 & pct>=46
replace receive = 1 if yr_month==202006 & stategr==2 & pct>=8
replace receive = 1 if yr_month==202006 & stategr==3 & pct>=2

replace PUC = 0 if receive==0
replace PUA = 0 if receive==0
replace RUI = 0 if receive==0

** Sum of UI benefits
gen PUCA=PUC+PUA
gen UIPUCA=RUI+PUC+PUA


** check whether the imputed benefit amount at the state-group level matches the total amount of UI benefits by admin. data 
bysort yr_month stategr: egen stgr_wgt = sum(wtfinl) 
sum stgr_totwgt stgr_wgt
foreach prgm in UIPUCA {
bysort yr_month asecflag serial famnum: egen stfam_`prgm' = sum(`prgm') 
gen stpc_`prgm' = stfam_`prgm'/fam_size
bysort yr_month stategr: egen stgr_tot`prgm' = sum((stgr_totwgt/stgr_wgt)*wtfinl*stpc_`prgm') 
}
bysort yr_month stategr: sum stgr_ccap stgr_totUIPUCA [w=wtfinl]

********************************************************************************
*** Table 3: Poverty Rates with and without COVID19 related Government Payments 
** calculate the aggregated simulated receipt/eligible amounts
bysort yr_month: egen wgt = sum(wtfinl) 
foreach prgm in EIP PUC PUA RUI PUCA UIPUCA {
bysort yr_month asecflag serial famnum: egen fam_`prgm' = sum(`prgm') 
gen pc_`prgm' = fam_`prgm'/fam_size
bysort yr_month: egen tot`prgm' = sum((totwgt/wgt)*wtfinl*pc_`prgm') 
replace tot`prgm' = tot`prgm'/1000000000
}
** Counterfactual family income w/o EIP and UI payment
gen inc_wo_All = ifam_inc-fam_EIP-fam_UIPUCA 
gen inc_wo_EIPUCA = ifam_inc-fam_EIP-fam_PUCA
gen inc_wo_EIP = ifam_inc-fam_EIP 
gen inc_wo_UIPUCA = ifam_inc-fam_UIPUCA 
gen inc_wo_PUCA = ifam_inc-fam_PUCA 

* counterfactual poverty status w/o EIP and UI payment
* five income measures (<100%, <200%, <300%, and <500% poverty threshold)
gen pov1 = ifam_inc<pov_thresh
gen pova = ifam_inc<pov_thresh*2
gen povb = ifam_inc<pov_thresh*3
gen povc = ifam_inc<pov_thresh*5
foreach prgm in All EIPUCA EIP UIPUCA PUCA {
gen pov_wo_`prgm' = inc_wo_`prgm'<pov_thresh
gen pova_wo_`prgm' = inc_wo_`prgm'<pov_thresh*2
gen povb_wo_`prgm' = inc_wo_`prgm'<pov_thresh*3
gen povc_wo_`prgm' = inc_wo_`prgm'<pov_thresh*5
}


* Actual 
forvalues mo = 1/6 {
di `mo'
preserve
keep yr_month asecflag serial wtfinl pov* 
egen ym = group(yr_month)
keep if ym==`mo' & asecflag!=1 
local num = `mo'+5

** Counterfactual Poverty
reg pov1 [w=wtfinl], cluster(serial)
putexcel set T3.xlsx, modify
putexcel A`num' = yr_month
putexcel B`num' = _b[_cons]
putexcel C`num' = _se[_cons]

foreach p in a b c {   
putexcel set T3.xlsx, sheet(panel`p') modify
reg pov`p' [w=wtfinl], cluster(serial)
putexcel A`num' = yr_month
putexcel B`num' = _b[_cons]
putexcel C`num' = _se[_cons]
}
restore
}


* Counterfactual 
forvalues mo = 1/6 {
di `mo'
preserve
keep yr_month asecflag serial wtfinl pov* 
egen ym = group(yr_month)
keep if ym==`mo'
local num = `mo'+5

** Counterfactual Poverty
putexcel set T3.xlsx, modify
reg pov_wo_All [w=wtfinl], cluster(serial)
putexcel D`num' = _b[_cons]
putexcel E`num' = _se[_cons]
reg pov_wo_EIPUCA [w=wtfinl], cluster(serial)
putexcel F`num' = _b[_cons]
putexcel G`num' = _se[_cons]
reg pov_wo_EIP [w=wtfinl], cluster(serial)
putexcel H`num' = _b[_cons]
putexcel I`num' = _se[_cons]
reg pov_wo_UIPUCA [w=wtfinl], cluster(serial)
putexcel J`num' = _b[_cons]
putexcel K`num' = _se[_cons]
reg pov_wo_PUCA [w=wtfinl], cluster(serial)
putexcel L`num' = _b[_cons]
putexcel M`num' = _se[_cons]

foreach p in a b c {   
putexcel set T3.xlsx, sheet(panel`p') modify
reg pov`p'_wo_All [w=wtfinl], cluster(serial)
putexcel D`num' = _b[_cons]
putexcel E`num' = _se[_cons]
reg pov`p'_wo_EIPUCA [w=wtfinl], cluster(serial)
putexcel F`num' = _b[_cons]
putexcel G`num' = _se[_cons]
reg pov`p'_wo_EIP [w=wtfinl], cluster(serial)
putexcel H`num' = _b[_cons]
putexcel I`num' = _se[_cons]
reg pov`p'_wo_UIPUCA [w=wtfinl], cluster(serial)
putexcel J`num' = _b[_cons]
putexcel K`num' = _se[_cons]
reg pov`p'_wo_PUCA [w=wtfinl], cluster(serial)
putexcel L`num' = _b[_cons]
putexcel M`num' = _se[_cons]
}
restore
}

preserve
clear
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\T3.xlsx", sheet("Sheet1")
xpose, clear 
gen v7 = v6-v1 if _n!=1
forvalues n = 3(2)13 {
    replace v7 = sqrt(v1^2+v6^2) if _n==`n'
	}
forvalues n = 1/3 {
    replace v`n' = . if _n>=4
}
tempfile main
save `main'
restore

foreach p in a b c {
preserve
clear
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\T3.xlsx", sheet("panel`p'")
xpose, clear 
gen v7 = v6-v1 if _n!=1
forvalues n = 3(2)13 {
    replace v7 = sqrt(v1^2+v6^2) if _n==`n'
	}
forvalues n = 1/3 {
    replace v`n' = . if _n>=4
}
tempfile panel`p'
save `panel`p''
restore
}

preserve
clear
use `main'
append using `panela'
append using `panelb'
append using `panelc'
export excel Table3.xlsx, keepcellfmt replace
restore


********************************************************************************
* Table 4.1 Imputed Cumulative Stimulus and UI Payments (billion $) 


preserve 
keep if yr_month>=202004
collapse totEIP totPUA totRUI totPUC totUIPUCA, by(yr_month)  
foreach var in EIP PUA RUI PUC UIPUCA {
    rename  tot`var' `var'b
    }
export excel T4.xlsx, keepcellfmt firstrow(var) replace
restore 


* Table 4.2 Person Receipt Rate (national level)

gen EIP_receipt = EIP_elig==1 & EIP>0
gen UI_receipt = UI_elig==1 & receive==1
gen PUA_receipt = PUA_elig==1 & receive==1
gen RUI_receipt = RUI_elig==1 & receive==1

forvalues m = 4/6 {
    preserve
	keep if yr_month==20200`m'
	local n = `m'-2
	putexcel set T4.xlsx, sheet("Sheet2") modify 
	putexcel A1 = "yr_month"
	putexcel B1 = "EIPc"
	putexcel C1 = "PUAc"
	putexcel D1 = "RUIc"
	putexcel E1 = "PUCc"
	putexcel F1 = "UIPUCAc"
    sum EIP_receipt [w=wtfinl] if EIP_elig==1 
	putexcel A`n' = yr_month
    putexcel B`n' = `r(mean)'
	sum PUA_receipt [w=wtfinl] if OPUA_elig==1 
    putexcel C`n' = `r(mean)'
	sum RUI_receipt [w=wtfinl] if RUI_elig==1 
    putexcel D`n' = `r(mean)'
	sum UI_receipt [w=wtfinl] if OUI_elig==1 
    putexcel E`n' = `r(mean)'
	sum UI_receipt [w=wtfinl] if OUI_elig==1 
    putexcel F`n' = `r(mean)'
	restore
	}


********************************************************************************
* Table 5. Person Receipt rates (state-group level)
** calculate the fraction of eligibles that receive the benefits (state group level)
preserve
keep if yr_month>=202004 & OPUA_elig==1
collapse PUA_receipt [w=wtfinl], by (yr_month stategr)
tempfile PUA
save `PUA'
restore

preserve
keep if yr_month>=202004 & RUI_elig==1
collapse RUI_receipt [w=wtfinl], by (yr_month stategr)
merge 1:1 yr_month stategr using `PUA'
drop _merge
order yr_month stategr PUA* RUI*
rename stategr tercile 
rename PUA_receipt receipt_rate1 
rename RUI_receipt receipt_rate2
gen i = _n
reshape long receipt_rate, i(i) j(UItype)
drop i
sort yr_month UItype tercile 
order yr_month UItype tercile receipt_rate
export excel Table5.xlsx, keepcellfmt firstrow(var) replace
restore

** calculate the fraction of Balck population (state group level)
//gen black = race==200
//bysort yr_month stategr: sum black [w=wtfinl] // use June number. page 16 in the draft


********************************************************************************
** Appendix Table 11. Comparison of the actual and imputed average WBA
bysort yr_month asecflag: sum AWB estben [w=wtfinl] if RUI_elig==1
preserve
keep if yr_month>=202004 & RUI_elig==1 
collapse AWB estben [w=wtfinl], by (yr_month)
gen Ratio = estben/AWB-1
rename AWB Actual_WBA
rename estben Imputed_WBA
rename Ratio Imputed_Actual_Ratio
export excel App.T11.xlsx, keepcellfmt firstrow(var) replace
restore


/*
** Comparison of the actual and imputed average WBA at the state level
preserve
bysort yr_month asecflag statefip: sum AWB estben [w=wtfinl] if RUI_elig==1
keep if yr_month>=202004 & RUI_elig==1 & asecflag==.
collapse AWB estben [w=wtfinl], by (yr_month statefip)
list
restore
*/

********************************************************************************
** AppT14: Fraction UI eligibles
preserve
bysort yr_month: sum OUI_elig OPUA_elig RUI_elig [w=wtfinl] if age>=16
keep if yr_month>=202004 & age>=16
gen cat = 0
replace cat = 1 if (empstat==21|empstat==22) & (classwkr==13|classwkr==14)
replace cat = 2 if whyabsnt==6
replace cat = 3 if whyabsnt==7
replace cat = 4 if whyabsnt==8
replace cat = 5 if whyabsnt==15
replace cat = 6 if wnlook==1
replace cat = 7 if wnlook==2
replace cat = 8 if wnlook==7
replace cat = 9 if wnlook==6
replace cat = 10 if wnlook==11
replace cat = 11 if empstat==22 & (classwkr!=13 & classwkr!=14) & wkstat == 50 
replace cat = 12 if empstat==22 & (classwkr!=13 & classwkr!=14) & wkstat == 60 
replace cat = 13 if empstat==21 & (classwkr!=13 & classwkr!=14) & wkstat == 50 
replace cat = 14 if empstat==21 & (classwkr!=13 & classwkr!=14) & wkstat == 60 
//drop if cat==0
replace wtfinl = round(wtfinl)
bysort yr_month: tab  cat [w=wtfinl] if age>=16

tabulate cat, generate(cat_)
//bysort yr_month: sum catid [w=wtfinl]
collapse cat_* [w=wtfinl], by (yr_month)
drop cat_1
rename cat_2 PUA_selfemp
rename cat_3 PUA_absent_health
rename cat_4 PUA_absent_family
rename cat_5 PUA_absent_childcare
rename cat_6 PUA_absent_other
rename cat_7 PUA_nosearch_nojob
rename cat_8 PUA_nosearch_nofind
rename cat_9 PUA_nosearch_family
rename cat_10 PUA_nosearch_childcare
rename cat_11 PUA_nosearch_other
rename cat_12 RUI_new_full
rename cat_13 RUI_new_part
rename cat_14 RUI_exp_full
rename cat_15 RUI_exp_part
xpose, clear varname
order _varname
export excel App.T14.xlsx, keepcellfmt firstrow(var) replace
restore


********************************************************************************
* Extra Table 1: Replacement rate 
gen totUI = PUC+PUA+RUI
gen replace_rate = 0
replace replace_rate = (totUI/min(durunemp,4))/pred_earn if yr_month==202004 & UI_receipt==1
replace replace_rate = (totUI/min(durunemp,8))/pred_earn if yr_month==202005 & UI_receipt==1
replace replace_rate = (totUI/min(durunemp,12))/pred_earn if yr_month==202006 & UI_receipt==1

* comparing to the cutoff
gen rr100 = replace_rate>1
gen rr150 = replace_rate>1.5
gen rr200 = replace_rate>2

preserve
keep if yr_month>=202004
keep if UI_receipt==1
collapse rr* [w=wtfinl], by (yr_month RUI_receipt)
sort RUI_receipt yr_month
order RUI_receipt yr_month
tempfile rr
save `rr'
restore

preserve
keep if yr_month>=202004
keep if UI_receipt==1
collapse rr* [w=wtfinl], by (yr_month)
append using `rr'
order RUI_receipt
export excel Extra1.xlsx, keepcellfmt firstrow(var) replace
restore

********************************************************************************
**********    Table 4: Calculate Simulated Eligible Amount          ************ 

clear
cd "C:\Users\worri\Dropbox\Poverty\COVID19"
use Monthly_Poverty_0520.dta 
gen yr_month=year*100+month   
keep if yr_month>=202001

drop month hflag qinc* qoinc* cpsid   
gen famnum = ftype==1|ftype==3 
sort yr_month asecflag serial ftype pernum
bysort yr_month asecflag serial: egen tot_subfam = sum(ftype==4 & famrel==1)
sort yr_month asecflag serial ftype famrel
bysort yr_month asecflag serial ftype: gen subfamnum = sum(famrel==1)
replace famnum= subfamnum+1 if ftype==4 
drop subfamnum 
bysort yr_month asecflag serial: gen non_fam = sum(famrel==0) 
replace famnum= non_fam+4 if famrel==0  
drop non_fam
bysort yr_month asecflag serial famnum: gen fam_size = _N
bysort yr_month asecflag serial famnum: egen nchild = sum(age<18 & relate>=301)
bysort yr_month asecflag serial famnum: egen head_elderly = max(age>=65 & relate==101)
replace nchild=nchild-1 if fam_size==nchild


* Combine with the Census poverty threshold data
merge m:1 year fam_size nchild head_elderly using pov_thresh
keep if _merge==3
drop _merge

* merge with the imputed family income data
merge m:1 yr_month asecflag serial using impute
keep if yr_month>=202001
drop _merge

********************************************************************************
* demographic variables 
gen male = sex==1
drop sex  
** merge with the state UI parameter data
merge m:1 yr_month statefip using C:\Users\worri\Dropbox\Poverty\COVID19\st_UI.dta 
drop _merge

********************************************************************************
** estimate population at the national and the state-group level
bysort yr_month: egen totwgt = sum(wtfinl) 
bysort yr_month stategr: egen stgr_totwgt = sum(wtfinl) 

*** Restrict the sample to the householder's family 
bysort yr_month asecflag serial famnum: egen hfam = sum(relate==101) 
keep if hfam==1

*** Restrict the sample to individuals with non-imputed income 
gen miss_inc = (faminc>=995 & faminc<=999)|(qfaminc>0 & qfaminc!=.)
keep if miss_inc!=1

bysort yr_month asecflag serial famnum: egen num_adult = sum(age>=18|famrel==1) 
bysort yr_month asecflag serial famnum: egen num_fam = sum(famrel==0|famrel==1)
sort yr_month asecflag serial famnum ftype famrel
bysort yr_month asecflag serial famnum: gen subfam = sum(famrel==0|famrel==1)
bysort yr_month asecflag serial famnum subfam: gen nspouse = sum(famrel==2)            // head's tax unit
replace subfam=2 if subfam==3 & famrel==2 & nspouse==2
bysort yr_month asecflag serial famnum subfam: egen n_adult = sum(age>=18|famrel==0|famrel==1)       
gen sfam_inc = ifam_inc/(num_adult)*n_adult


gen tunit = ((famrel==3|famrel==4) & age>=24)
sort yr_month asecflag serial famnum subfam tunit
bysort yr_month asecflag serial famnum subfam: gen taxunit = sum(tunit)
replace taxunit = taxunit+1
bysort yr_month asecflag serial famnum subfam taxunit: gen tu_size = _N
bysort yr_month asecflag serial famnum subfam taxunit: egen tu_n_adult = sum(age>=18|famrel==0|famrel==1) 
bysort yr_month asecflag serial famnum subfam taxunit: egen tu_n_nodep = sum(famrel==0|famrel==1|famrel==2|age>=24)
gen tu_n_dep = tu_size-tu_n_nodep

gen tfam_inc = (sfam_inc/n_adult)*tu_n_adult 
gen EIP = 0
replace EIP = 1200*tu_n_nodep+500*tu_n_dep if tfam_inc<=75000*tu_n_nodep
replace EIP = 1200*tu_n_nodep+500*tu_n_dep-0.05*(tfam_inc-75000*tu_n_nodep) if tfam_inc>75000*tu_n_nodep & tfam_inc<99000*tu_n_nodep
replace EIP = EIP/tu_size
gen EIP_elig = EIP>0
replace EIP = 0 if yr_month<=202003

preserve
import delimited using "State UI benefit calculations", clear
drop if statefip==.
rename ïstate state
label var state "State"
tempfile UIrule
save `UIrule'
restore

merge m:1 statefip using `UIrule'
drop _merge

gen PUA_unemp = (empstat==21|empstat==22) & (classwkr==13|classwkr==14) 
gen PUA_nounemp = (((whyabsnt>=6 & whyabsnt<=8)|whyabsnt==15)|(wnlook==1|wnlook==2|wnlook==6|wnlook==7|wnlook==11)) 
gen PUA_elig = (PUA_unemp==1|PUA_nounemp==1)
gen RUI_elig = (empstat==21|empstat==22) & PUA_elig==0
 
bysort yr_month: sum durunemp [w=wtfinl] if durunemp>=0 & durunemp!=999
replace durunemp = 8 if PUA_nounemp==1 & yr_month==202004
replace durunemp = 11 if PUA_nounemp==1 & yr_month==202005
replace durunemp = 13 if PUA_nounemp==1 & yr_month==202006
gen UI_elig = durunemp>=0 & durunemp!=999
bysort yr_month: sum PUA_elig RUI_elig [w=wtfinl] if UI_elig==1
gen OUI_elig = UI_elig
gen OPUA_elig = PUA_elig

gen earnsample = mish==4|mish==8

* age
gen age_gr = 0
replace age_gr = 1 if age<18
replace age_gr = 2 if age>=18 & age<30
replace age_gr = 3 if age>=31 & age<45
replace age_gr = 4 if age>=46 & age<65
replace age_gr = 5 if age>=65

* edu
gen educa = 0
replace educa=1 if educ<73
replace educa=2 if educ==73   
replace educa=3 if educ>=74 & educ<111  
replace educa=4 if educ>=111 & educ!=.

** race/ethnicity
gen racethn = 0
replace racethn = 1 if hispan==0 & race==100 // white
replace racethn = 2 if hispan==0 & race==200 // black
replace racethn = 3 if hispan==0 & race!=100 & race!=200  // other
replace racethn = 4 if hispan!=0  // hispan

** occupation (9)
gen occ_ma = (occ>=10 & occ<=3655)    // managerial, professional, technical
gen occ_se = (occ>=3700 & occ<=4655)  // service 
gen occ_sa =  (occ>=4700 & occ<=4965)   // Sales
gen occ_of = (occ>=5000 & occ<=5940)   // Office/Admin. Support
gen occ_fa = (occ>=6005 & occ<=6130)     // Farming, Fishing, and Forestry (Natural resources)
gen occ_co = (occ>=6200 & occ<=7640)   // Construction/Repair
gen occ_pr = (occ>=7700 & occ<=9760)   // Production/Transportation
gen occ_mi = (occ>=9800 & occ<=9840) // Military
gen occ_no = occ==0                  // Nonemployed

* Industry Recodes (10)
gen ind_ag = (ind>=170 & ind<=490)   //agriculture, mining 
gen ind_co = (ind>=770 & ind<=3990)  //construction, manufacturing 
gen ind_wh = (ind>=4070 & ind<=5790) //retale trade
gen ind_tr = (ind>=6070 & ind<=6390)|(ind>=570 & ind<=690) //Transportation and utilities
gen ind_in = (ind>=6470 & ind<=6780) //information
gen ind_fi = (ind>=6870 & ind<=7190) //finance
gen ind_pr = (ind>=7270 & ind<=7790) //Professional and business services
gen ind_ed = (ind>=7860 & ind<=9590) //other service, public admin
gen ind_ar = (ind>=9670 & ind<=9870) //Armed Forces
gen ind_no = (ind==9920)             //nonemployed

replace earnweek=. if earnweek==9999.99
reg earnweek age male i.educa i.racethn ind_* occ_* i.statefip i.yr_month 
predict pred_earn
replace pred_earn=pred_earn*0.875 if RUI_elig==1
replace pred_earn=pred_earn*0.836 if PUA_elig==1
replace pred_earn=36.25 if pred_earn<36.25 
replace pred_earn=0 if OUI_elig==0
gen qtrearn=pred_earn*13
keep if mish==1|mish==5

merge 1:1 yr_month asecflag serial pernum using fixedwgt.dta
replace wtfinl = fixedwgt if yr_month>=202003

* estimate benefits for regular UI recipients 
gen estben=min(maxben,max(minben,qtrearn*replacement_rate))
label var estben "Estimated Weekly UI benefit"
* estimate benefits for PUA UI recipients 
replace estben = max(estben,pua_min) if PUA_elig==1
* zero benefits to those who are not eligible for UI
replace estben = 0 if UI_elig==0


********************************************************************************
*** Calculate the cumulative UI benefits by multiplying WBA by #weeks unemployed   
gen PUC = 0
gen PUA = 0
gen RUI = 0
** PUC 
replace PUC = 600*min(durunemp,4) if durunemp!=999 & yr_month==202004
replace PUC = 600*min(durunemp,8) if durunemp!=999 & yr_month==202005
replace PUC = 600*min(durunemp,12) if durunemp!=999 & yr_month==202006

** PUA  
replace PUA = estben*min(durunemp,13) if durunemp!=999 & PUA_elig==1 & yr_month==202004
replace PUA = estben*min(durunemp,17) if durunemp!=999 & PUA_elig==1 & yr_month==202005
replace PUA = estben*min(durunemp,21) if durunemp!=999 & PUA_elig==1 & yr_month==202006

** Regular UI 
replace RUI = estben*min(durunemp,4) if durunemp!=999 & RUI_elig==1 & yr_month==202004
replace RUI = estben*min(durunemp,8) if durunemp!=999 & RUI_elig==1 & yr_month==202005
replace RUI = estben*min(durunemp,12) if durunemp!=999 & RUI_elig==1 & yr_month==202006

* for Oregon, there is one week waiting period for UI 
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=3 & RUI_elig==1 & yr_month==202004 & statefip==41
replace RUI = estben*3 if durunemp>=4 & durunemp!=999 & RUI_elig==1 & yr_month==202004 & statefip==41
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=7 & RUI_elig==1 & yr_month==202005 & statefip==41
replace RUI = estben*7 if durunemp>=8 & durunemp!=999 & RUI_elig==1 & yr_month==202005 & statefip==41
replace RUI = estben*(max(durunemp-1,0)) if durunemp<=11 & RUI_elig==1 & yr_month==202006 & statefip==41
replace RUI = estben*11 if durunemp>=12 & durunemp!=999 & RUI_elig==1 & yr_month==202006 & statefip==41

** Calculate the aggregate benefit amount 
gen UIPUCA=RUI+PUC+PUA


********************************************************************************
** Table 4. Simulated Eligible Amount

** total payment at the national level
bysort yr_month: egen wgt = sum(wtfinl) 
sum totwgt wgt
foreach prgm in EIP PUA RUI PUC UIPUCA {
bysort yr_month asecflag serial famnum: egen fam_`prgm' = sum(`prgm') 
gen pc_`prgm' = fam_`prgm'/fam_size
bysort yr_month: egen tot`prgm' = sum((totwgt/wgt)*wtfinl*pc_`prgm') 
replace tot`prgm' = tot`prgm'/1000000000
}
preserve 
keep if yr_month>=202004
collapse totEIP totPUA totRUI totPUC totUIPUCA, by(yr_month) 
list
foreach var in EIP PUA RUI PUC UIPUCA {
    rename  tot`var' `var'a
    }
export excel T4.xlsx, sheet("Sheet3") keepcellfmt firstrow(var) 
restore



clear
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\T4.xlsx", firstrow  sheet("Sheet1")
sum
forvalues n = 2/3 {
preserve
clear
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\T4.xlsx", firstrow sheet("Sheet`n'")
tempfile panel`n'
save `panel`n''
restore
merge 1:1 yr_month using `panel`n''
drop _merge
} 
reshape long EIP PUA RUI PUC UIPUCA, i(yr_month) j(rotation)  string
reshape wide EIP PUA RUI PUC UIPUCA, i(rotation) j(yr_month) 
drop rotation
xpose, clear v
order _varname 
rename v1 sim_elig
rename v2 sim_receipt
rename v3 person_receipt_rate
gen dollar_receipt_rate = sim_receipt/sim_elig
gen admin_payment = .
replace admin_payment= 160 if _n==1
replace admin_payment=32.4 if _n==5
replace admin_payment= 259 if _n==6
replace admin_payment= 122 if _n==10
replace admin_payment= 267 if _n==11
replace admin_payment= 224 if _n==15
order _varname sim_elig admin_payment sim_receipt dollar_receipt_rate person_receipt_rate
export excel Table4.xlsx, keepcellfmt firstrow(var) replace

/*
********************************************************************************
********************************************************************************
** calculate adjustment factors to scale down imputed earnings for PUA/RUI eligibles (see line 1065)

clear
cd "C:\Users\worri\Dropbox\Poverty\COVID19"
use Monthly_Poverty_0520.dta 

gen yr_month=year*100+month   
keep if yr_month>=201904
drop if cpsid==0

** drop irrelevant variables
drop month hflag qinc* qoinc* cpsid   

* demographic variables 
gen male = sex==1
drop sex  


** merge with the state UI parameter data
merge m:1 yr_month statefip using C:\Users\worri\Dropbox\Poverty\COVID19\st_UI.dta 
drop _merge


** Define the PUA and regular UI eligibles 
* 1) the unemployed who were self-employed 
gen PUA_unemp = (empstat==21|empstat==22) & (classwkr==13|classwkr==14) 
* 2) the employed who is absent from work due to health/family responsibilies/other reasons
* 3) people who want a job but did not look for a job for 4 weeks b/c no job available/couldn't find a job/family responsibilies/other reasons 
gen PUA_nounemp = (((whyabsnt>=6 & whyabsnt<=8)|whyabsnt==15)|(wnlook==1|wnlook==2|wnlook==6|wnlook==7|wnlook==9|wnlook==11)) 
* PUA eligibles
gen PUA_elig = (PUA_unemp==1|PUA_nounemp==1)
* Regular UI eligibles (the unemployed who was not self-employed)
gen RUI_elig = (empstat==21|empstat==22) & PUA_elig==0
* drop four observations in Jan/Feb 2020 w/o information on the duration of unemployment
drop if (empstat==21|empstat==22) & durunemp==999
 
* impute the unemployment duration for those who are not unemployed (assign the average unemployment duration) 
bysort yr_month: sum durunemp [w=wtfinl] if durunemp>=0 & durunemp!=999
replace durunemp = 8 if PUA_nounemp==1 & yr_month==202004
replace durunemp = 11 if PUA_nounemp==1 & yr_month==202005
replace durunemp = 13 if PUA_nounemp==1 & yr_month==202006
* UI eligibles (PUA or regular UI)
gen UI_elig = durunemp>=0 & durunemp!=999
bysort yr_month: sum PUA_elig RUI_elig [w=wtfinl] if UI_elig==1

* age
gen age_gr = 0
replace age_gr = 1 if age<18
replace age_gr = 2 if age>=18 & age<30
replace age_gr = 3 if age>=31 & age<45
replace age_gr = 4 if age>=46 & age<65
replace age_gr = 5 if age>=65

* edu
gen educa = 0
replace educa=1 if educ<73
replace educa=2 if educ==73   
replace educa=3 if educ>=74 & educ<111  
replace educa=4 if educ>=111 & educ!=.

** race/ethnicity
gen racethn = 0
replace racethn = 1 if hispan==0 & race==100 // white
replace racethn = 2 if hispan==0 & race==200 // black
replace racethn = 3 if hispan==0 & race!=100 & race!=200  // other
replace racethn = 4 if hispan!=0  // hispan

** occupation (9)
gen occ_ma = (occ>=10 & occ<=3655)    // managerial, professional, technical
gen occ_se = (occ>=3700 & occ<=4655)  // service 
gen occ_sa =  (occ>=4700 & occ<=4965)   // Sales
gen occ_of = (occ>=5000 & occ<=5940)   // Office/Admin. Support
gen occ_fa = (occ>=6005 & occ<=6130)     // Farming, Fishing, and Forestry (Natural resources)
gen occ_co = (occ>=6200 & occ<=7640)   // Construction/Repair
gen occ_pr = (occ>=7700 & occ<=9760)   // Production/Transportation
gen occ_mi = (occ>=9800 & occ<=9840) // Military
gen occ_no = occ==0                  // Nonemployed

* Industry Recodes (10)
gen ind_ag = (ind>=170 & ind<=490)   //agriculture, mining 
gen ind_co = (ind>=770 & ind<=3990)  //construction, manufacturing 
gen ind_wh = (ind>=4070 & ind<=5790) //retale trade
gen ind_tr = (ind>=6070 & ind<=6390)|(ind>=570 & ind<=690) //Transportation and utilities
gen ind_in = (ind>=6470 & ind<=6780) //information
gen ind_fi = (ind>=6870 & ind<=7190) //finance
gen ind_pr = (ind>=7270 & ind<=7790) //Professional and business services
gen ind_ed = (ind>=7860 & ind<=9590) //other service, public admin
gen ind_ar = (ind>=9670 & ind<=9870) //Armed Forces
gen ind_no = (ind==9920)             //nonemployed


********************************************************************************
****  Estimate the earnings ratio between PUA and Regular UI eligibles    
gen rotation = mish==4
replace rotation = 2 if mish==5
//replace rotation = 3 if mish>=1 & mish<=3
drop if rotation==0

* change the data structure as panel structure 
keep yr_month cpsidp serial age male educa racethn ind_* occ_* statefip wtfinl earnwt rotation earnweek PUA_elig RUI_elig UI_elig
reshape wide yr_month serial age male educa racethn ind_* occ_* statefip wtfinl earnwt earnweek PUA_elig RUI_elig UI_elig, i(cpsidp) j(rotation)

sort cpsidp yr_month* serial* age* male* educa* racethn* ind_* occ_* statefip* wtfinl* earnwt* earnweek* PUA_elig* RUI_elig* UI_elig* 
order cpsidp yr_month* serial* age* male* educa* racethn* ind_* occ_* statefip* wtfinl* earnwt* earnweek* PUA_elig* RUI_elig* UI_elig* 


********************************************************************************
** Sample restriction
** 1) individuals who did their 5th month (4th month) interview in April-June 2020 (July-Spetember 2019) 
keep if yr_month1!=. & yr_month2>=202004 & yr_month2<=202006

** 2) individuals who had earnings in 4th month interview
replace earnweek1=. if earnweek1==9999.99
keep if earnweek1!=.

** 3) individuals who had positive weight in both 4th and 5th month interviews
gen allpwgt = wtfinl1!=0 & wtfinl2!=0 
sum allpwgt [w=wtfinl1]
keep if allpwgt==1

** average earnings among UI non-eligible
sum earnweek1 [w=earnwt1] if UI_elig2==0
* 1098.8
reg earnweek1 PUA_elig2 RUI_elig2 age1 male1 i.educa1 i.racethn1 ind_*1 occ_*1 i.statefip1 i.yr_month1
* coefficint on PUA_eilg2: -180.2
* coefficient on RUI_eilg2: -137.8
* earnings ratio PUA_elig2/UI_nonelig2: 0.8360 =  918.6/1098.8
* earnings ratio RUI_eilg2/UI_nonelig2: 0.8746 =  961/1098.8
*/
